<?php
namespace controller;

use \lib\db as db;

class Games extends \Pawn\Controller
{
	public function _construct($skip=false)
	{
		if(!$skip && !$this->total_games)
		{
			return $this->redirect('Users', 'leaderboard');
		}
		
		\lib\Lang::load('games');
		
		if(isset($_GET['r']))
		{
			setcookie('referer', $_GET['r'], 0, WEB);
		}
	}
	
	# Categories
	public $categories = array
	(
		'action'	=>	'Action',
		'adventure'	=>	'Adventure',
		'board'		=>	'Board game',
		'casino'	=>	'Casino',
		'driving'	=>	'Driving',
		'dress_up'	=>	'Dress Up',
		'fighting'	=>	'Fighting',
		'puzzle'	=>	'Puzzles',
		'customize'	=>	'Pimp my / Customize',
		'shooting'	=>	'Shooting',
		'sports'	=>	'Sports',
		'other'		=>	'Other',
		'strategy'	=>	'Strategy',
		'education'	=>	'Education',
		'rhythm'	=>	'Rhythm',
		'jigsaw'	=>	'Jigsaw / Slider Puzzles'
	);
	
	# Browse
	public function browse($category='all', $order='date', $dir='desc', $page=1)
	{
		$query = '';
		
		// Category
		$instances = \model\Category::get('WHERE enabled=1 AND games>0')->fetchAll();
		
		$categories = array
		(
			'all'	=>	\lib\Lang::get('categories', 'all')
		);
		
		if(count($instances) > 1)
		{
			foreach($instances as $instance)
			{
				$categories[array_search($instance->name, $this->categories)] = $instance;
			}
		}
		
		if(isset($categories[$category]) && $category != 'all')
		{
			$query .= 'WHERE category_id='.$categories[$category]->id.' ';
			
			$games_total = $categories[$category]->games;
		} else
		{
			$category = 'all';
			
			$games_total = \lib\db::execute('SELECT SUM(games) FROM '.\model\Category::table_name().' WHERE enabled=1')->fetchColumn();
		}
		
		// Ordering
		$ordering = array
		(
			'date'			=>	'id',
			'title'			=>	'name',
			'rating'		=>	'rating',
			'plays'			=>	'plays',
			'favourites'	=>	'favourites',
			'comments'		=>	'comments'
		);
		
		if(!isset($ordering[$order]))
		{
			$order = 'date';
		}
		
		$query .= 'ORDER BY `'.$ordering[$order].'` ';
		
		foreach($ordering as $key=>$value)
		{
			$ordering[$key] = \lib\Lang::get('ordering', $key);
		}
		
		// Direction
		$dir = ($dir == 'asc')?'asc':'desc';
		
		$query .= $dir.' ';
		
		// Page
		$pageSize = 14;
		
		if(\Pawn::setting('ad_200x90'))
		{
			$pageSize--;
		}
		
		$pages = ceil($games_total/$pageSize);
		
		if($page < 1)
		{
			$page = 1;
		} elseif($page > $pages)
		{
			$page = $pages;
		}
		
		$query .= ' LIMIT '.(($page-1)*$pageSize).','.$pageSize;
		
		// Games
		$games = \model\Game::get($query)->fetchAll();
		
		// Render
		return $this->render('games/browse.html', array
		(
			'category'		=>	$category,
			'order'			=>	$order,
			'dir'			=>	$dir,
			'page'			=>	$page,
			
			'categories'	=>	$categories,
			'ordering'		=>	$ordering,
			'pages'			=>	$pages,
			'path'			=>	\Pawn::path('Games', 'browse', array('category'=>'{category}', 'order'=>'{order}', 'dir'=>'{dir}', 'page'=>'{page}')),
			
			'games'			=>	$games,
			'featured'		=>	$this->featured(),
			'spotlight'		=>	$this->spotlight()
		));
	}
	
	# Featured
	private function featured()
	{
		return \model\Feature::get('WHERE image="" ORDER BY ts_create DESC')->fetchAll();
	}
	
	# Spotlight
	private function spotlight()
	{
		return \model\Feature::get('WHERE image!="" ORDER BY ts_edit DESC')->fetchAll();
	}
	
	# Search
	public function search($input, $page)
	{
		// Input
		$words = explode(' ', urldecode($input));
		
		$query = '';
		
		foreach($words as $i=>$word)
		{
			if(strlen($word) == 1)
			{
				continue;
			}
			
			if($query)
			{
				$query .= ' OR ';
			}
			
			$query .= 'name LIKE '.\lib\db::quote('%'.$word.'%').' OR keywords LIKE '.\lib\db::quote('%'.$word.'%');
		}
		
		if($query)
		{
			$games = \model\Game::get('WHERE '.$query)->fetchAll();
		} else
		{
			$games = array();
		}
		
		// Page
		$pageSize = 14;
		
		if(\Pawn::setting('ad_200x90'))
		{
			$pageSize--;
		}
		
		$total = count($games);
		$pages = ceil($total/$pageSize);
		
		if($page < 1)
		{
			$page = 1;
		} elseif($page > $pages)
		{
			$page = $pages;
		}
		
		$games = array_slice($games, ($page-1)*$pageSize, $pageSize);
		
		// Render
		return $this->render('games/search.html', array
		(
			'input'		=>	$input,
			'page'		=>	$page,
			
			'total'		=>	$total,
			'pages'		=>	$pages,
			
			'games'		=>	$games,
			'featured'	=>	$this->featured(),
			'spotlight'	=>	$this->spotlight()
		));
	}
	
	# Order
	private $order, $dir;
	
	public function order($a, $b)
	{
		$a = $a->{$this->order};
		$b = $b->{$this->order};
		
		if($a == $b)
		{
			return 0;
		}
		
		return (($this->dir == 'desc')?1:-1)*(($a < $b)?1:-1);
	}
	
	# Recommended
	private function recommended($count=5, $game=false)
	{
		$games = array();
		$recommended = array();
		
		if($this->user->authenticated())
		{
			$favourites = db::query('SELECT `game_id` FROM `'.\model\Favourite::table_name().'` WHERE `user_id`='.$this->user->id)->fetchAll(\PDO::FETCH_COLUMN);
			
			foreach($favourites as $favourite)
			{
				$users = db::query('SELECT `user_id` FROM `'.\model\Favourite::table_name().'` WHERE `game_id`='.$favourite)->fetchAll(\PDO::FETCH_COLUMN);
				
				foreach($users as $user)
				{
					if($user == $this->user->id)
					{
						continue;
					}
					
					$user_favourites = db::query('SELECT `game_id` FROM `'.\model\Favourite::table_name().'` WHERE `user_id`='.$user)->fetchAll(\PDO::FETCH_COLUMN);
					
					foreach($user_favourites as $user_favourite)
					{
						if($game && $user_favourite == $game->id)
						{
							continue;
						}
				
						$games[$user_favourite] = isset($games[$user_favourite])?($games[$user_favourite]+1):1;
					}
				}
			}
			
			arsort($games);
			
			$games = array_slice(array_keys($games), 0, $count);
			
			foreach($games as $id)
			{
				$recommended[] = \model\Game::get('WHERE id=?', $id)->fetch();
			}
		}
		
		if(count($recommended) < $count)
		{
			$this->randomFillerGames($recommended, $games, $count);
		}
		
		return $recommended;
	}
	
	# Favourites
	public function favourites($user, &$order, &$dir, &$page)
	{
		// Games
		$games = array();
		
		// Favourites
		$favourites = \model\Favourite::get('WHERE user_id=?', $user->id);
		
		foreach($favourites as $favourite)
		{
			$games[$favourite->ts_create] = $favourite->game;
		}
		
		// Direction
		$dir = ($dir == 'asc')?'asc':'desc';
		
		// Ordering
		$ordering = array
		(
			'date'			=>	'id',
			'title'			=>	'name',
			'rating'		=>	'rating',
			'plays'			=>	'plays',
			'favourites'	=>	'favourites',
			'comments'		=>	'comments'
		);
		
		if(!isset($ordering[$order]) || $order == 'date')
		{
			$order = 'date';
			
			if($dir == 'asc')
			{
				ksort($games);
			} else
			{
				krsort($games);
			}
		} else
		{
			$this->dir = $dir;
			$this->order = $ordering[$order];
			
			usort($games, array($this, 'order'));
		}
		
		foreach($ordering as $key=>$value)
		{
			$ordering[$key] = \lib\Lang::get('ordering', $key);
		}
		
		// Page
		$pageSize = 6;
		
		if(\Pawn::setting('ad_200x90'))
		{
			$pageSize--;
		}
		
		$total = count($games);
		
		$from = ($page-1)*$pageSize+1;
		$to = $page*$pageSize;
		
		if($to > $total)
		{
			$to = $total;
		}
		
		$pages = ceil($total/$pageSize);
		
		if($page < 1)
		{
			$page = 1;
		} elseif($page > $pages)
		{
			$page = $pages;
		}
		
		$games = array_slice($games, ($page-1)*$pageSize, $pageSize);
		
		// Render
		return array
		(
			'order'			=>	$order,
			'dir'			=>	$dir,
			'page'			=>	$page,
			
			'ordering'		=>	$ordering,
			'pages'			=>	$pages,
			
			'from'			=>	$from,
			'to'			=>	$to,
			'total'			=>	$total,
			
			'favourites'	=>	$games
		);
	}
	
	# Play
	public function play($slug)
	{
		// Game
		$game = \model\Game::get('WHERE slug=?', $slug)->fetch();
		
		if(!$game)
		{
			return $this->not_found();
		}
		
		$game->plays++;
		
		if($this->user->authenticated())
		{
			$play = \model\Plays::get('WHERE user_id=? AND game_id=?', array($this->user->id, $game->id))->fetch();
			
			if(!$play)
			{
				$play = new \model\Plays(array
				(
					'user_id'	=>	$this->user->id,
					'game_id'	=>	$game->id,
					'times'		=>	1
				), true);
				
				$play->create();
				
				$this->user->awardPoints(1);
			} else
			{
				$play->times++;
			}
		}
		
		if($game->width > 938)
		{
			$width = 938;
			$height = $game->height*$width/$game->width;
		} else
		{
			$width = $game->width;
			$height = $game->height;
		}
		
		// Categories
		$categories = array();
		
		$identifiers = explode(',', $game->categories);
		
		foreach($identifiers as $id)
		{
			$instance = \model\Category::get('WHERE id=?', $id)->fetch();
			
			$categories[array_search($instance->name, $this->categories)] = $instance;
		}
		
		if(\Pawn::setting('Game.publisher_id') && $game->leaderboard_enabled)
		{
			// Leaderboard
			$leaderboard = array
			(
				'day'	=>	array(),
				'week'	=>	array(),
				'month'	=>	array(),
				'all'	=>	array()
			);
			
			if($order = db::query('SELECT `order` FROM `'.\model\Score::table_name().'` WHERE `game_id`=?', $game->id)->fetchColumn())
			{
				$time = time();
				$select = (($order == 'asc')?'MIN':'MAX').'(`score`)';
				
				$query = 'SELECT `user_id`, '.$select.' FROM `'.\model\Score::table_name().'` WHERE `game_id`=?';
				$group = ' GROUP BY `user_id` ORDER BY '.(($order == 'asc')?'MIN(`SCORE`) ASC':'MAX(`SCORE`) DESC').' LIMIT 10';
				
				$periods = array
				(
					'day'	=>	86400,
					'week'	=>	604800,
					'month'	=>	2629743,
					'all'	=>	0
				);
				
				foreach($periods as $period=>$length)
				{
					if($length)
					{
						$submissions = db::query($query.' AND `ts_create`>?'.$group, array($game->id, $time - $length))->fetchAll();
					} else
					{
						$submissions = db::query($query.$group, $game->id)->fetchAll();
					}
					
					foreach($submissions as $submission)
					{
						$leaderboard[$period][] = array
						(
							'user'	=>	\model\User::get('WHERE id=?', $submission['user_id'])->fetch(),
							'score'	=>	$submission[$select]
						);
					}
				}
			}
			
			// Places
			$places = $this->places($game->id, $order);
		} else
		{
			$leaderboard = false;
			$places = false;
		}
		
		// Similar
		$games = array();
		
		$rows = db::query('SELECT `user_id` FROM `'.\model\Favourite::table_name().'` WHERE `game_id`='.$game->id)->fetchAll(\PDO::FETCH_COLUMN);
		
		foreach($rows as $id)
		{
			$favourites = db::query('SELECT `game_id` FROM `'.\model\Favourite::table_name().'` WHERE `user_id`='.$id)->fetchAll(\PDO::FETCH_COLUMN);
			
			foreach($favourites as $favourite)
			{
				if($favourite == $game->id)
				{
					continue;
				}
				
				$games[$favourite] = isset($games[$favourite])?($games[$favourite]+1):1;
			}
		}
		
		$rows = db::query('SELECT `id`, `categories` FROM `'.\model\Game::table_name().'` WHERE `trash`=0 AND `category_id` IN ('.$game->categories.')')->fetchAll();
		
		foreach($rows as $row)
		{
			if($row['id'] == $game->id)
			{
				continue;
			}
			
			$values = explode(',', $row['categories']);
			
			foreach($values as $id)
			{
				if(in_array($id, $identifiers))
				{
					$games[$row['id']] = isset($games[$row['id']])?($games[$row['id']]+1):1;
				}
			}
		}
		
		arsort($games);
		
		$games = array_slice(array_keys($games), 0, 5);
		
		$similar = array();
		
		foreach($games as $id)
		{
			$similar[] = \model\Game::get('WHERE id=?', $id)->fetch();
		}
		
		if(count($similar) < 5)
		{
			$this->randomFillerGames($similar, $games);
		}
		
		// Recommended
		$recommended = $this->recommended();
		
		// Comments
		$write_comment = new \lib\Form('comment');
		$write_comment->field('message', 'textarea', array(), \lib\Lang::get('comment_placeholder'));
		
		$comments = \model\Comment::get('WHERE target="game" AND target_id=? ORDER BY ts_edit DESC LIMIT 50', $game->id)->fetchAll();
		
		// Render
		return $this->render('games/play.html', array
		(
			'game'			=>	$game,
			
			'rated'			=>	($this->user->authenticated() && $rating = \model\Rating::get('WHERE user_id=? AND target_id=? AND target="game"', array($this->user->id, $game->id))->fetch())?$rating->stars:0,
			'favourited'	=>	($this->user->authenticated() && \model\Favourite::get('WHERE user_id=? AND game_id=?', array($this->user->id, $game->id))->fetch())?true:false,
			
			'width'			=>	$width,
			'height'		=>	$height,
			
			'categories'	=>	$categories,
			
			'leaderboard'	=>	$leaderboard,
			'places'		=>	$places,
			
			'similar'		=>	$similar,
			'recommended'	=>	$recommended,
			
			'comments'		=>	$comments,
			'write_comment'	=>	$write_comment
		));
	}
	
	private function randomFillerGames(&$container, $games, $count=5)
	{
		$total = \model\Game::num_rows('WHERE `trash`=0');
		
		if($total <= $count)
		{
			$container = false;
		} else
		{
			while(count($container) < $count)
			{
				$index = mt_rand(0, $total - 1);
				
				$instance = \model\Game::get('WHERE `trash`=0 LIMIT '.$index.',1')->fetch();
				
				if($instance && !in_array($instance->id, $games))
				{
					$games[] = $instance->id;
					
					$container[] = $instance;
				}
			}
		}
	}
	
	public function rate($type, $id)
	{
		$stars = (int) $_POST['stars'];
		
		if($this->user->authenticated() && $stars >= 1 && $stars <= 10)
		{
			$target = false;
			
			switch($type)
			{
				case 'game':
					$target = \model\Game::get('WHERE id=?', $id)->fetch();
				break;
				case 'user':
					$target = \model\User::get('WHERE id=?', $id)->fetch();
				break;
			}
			
			if(!$target)
			{
				exit;
			}
			
			$total = \model\Rating::num_rows('WHERE target=? AND target_id=?', array($type, $id));
			$current = \model\Rating::get('WHERE user_id=? AND target=? AND target_id=?', array($this->user->id, $type, $id))->fetch();
			
			if($current)
			{
				$target->rating = ($target->rating*$total - $current->stars + $stars)/$total;
				$current->stars = $stars;
			} else
			{
				$rating = new \model\Rating(array
				(
					'user_id'	=>	$this->user->id,
					'target'	=>	$type,
					'target_id'	=>	$target->id,
					'stars'		=>	$stars
				), true);
				
				if($rating->create()->valid)
				{
					$this->user->awardPoints(1);
					
					$target->rating = ($target->rating*$total + $stars)/($total+1);
					
					if($type == 'user')
					{
						$target->awardPoints(1);
					}
				}
			}
			
			return $this->json($target->rating);
		}
	}
	
	public function favourite($id)
	{
		if($this->user->authenticated() && $game = \model\Game::get('WHERE id=?', $id)->fetch())
		{
			$favourited = \model\Favourite::get('WHERE user_id=? AND game_id=?', array($this->user->id, $id))->fetch();
			
			if($favourited)
			{
				$game->favourites--;
				
				$favourited->delete();
			} else
			{
				$game->favourites++;
				
				$favourite = new \model\Favourite(array
				(
					'user_id'	=>	$this->user->id,
					'game_id'	=>	$id
				), true);
				
				$favourite->create();
			}
			
			return $this->json(true);
		}
	}
	
	public function comment($type, $id)
	{
		if($this->user->authenticated() && isset($_POST['message']))
		{
			$target = false;
			
			switch($type)
			{
				case 'game':
					$target = \model\Game::get('WHERE id=?', $id)->fetch();
				break;
				case 'user':
					$target = \model\User::get('WHERE id=?', $id)->fetch();
				break;
			}
			
			if(!$target)
			{
				exit;
			}
			
			if($_POST['edit'] != '0')
			{
				if($comment = \model\Comment::get('WHERE id=?', $_POST['edit'])->fetch())
				{
					if($this->user->level >= 3 || $comment->user_id == $this->user->id)
					{
						$comment->ts_edit = time();
						$comment->message = $_POST['message'];
						
						$response = array($comment);
					}
				}
			} else
			{
				$response = \model\Comment::get('WHERE target=? AND target_id=? AND id>? ORDER BY ts_edit ASC', array($type, $id, $_POST['id']))->fetchAll();
				
				$comment = new \model\Comment(array
				(
					'user_id'	=>	$this->user->id,
					'target'	=>	$type,
					'target_id'	=>	$id,
					'message'	=>	$_POST['message']
				), true);
				
				if($comment->create()->valid)
				{
					if($type == 'game')
					{
						$target->comments++;
					}
				}
				
				$response[] = $comment;
			}
			
			$new_comment = $comment->id;
		} elseif($_POST['action'] == 'delete')
		{
			if($comment = \model\Comment::get('WHERE id=?', $_POST['id'])->fetch())
			{
				$comment->delete();
				
				return $this->json(true);
			}
		}
		
		$comments = array();
		
		foreach($response as $row)
		{
			$comments[] = array
			(
				'id'		=>	$row->id,
				'ts'		=>	$row->ts_create,
				'ago'		=>	\Pawn::time_ago($row->ts_edit),
				'user'		=>	$row->user->profile(),
				'avatar'	=>	$row->user->file('avatar'),
				'profile'	=>	\Pawn::path('Users', 'profile', array('name'=>$row->user->username)),
				'level'		=>	$row->user->exp_level(),
				'msg'		=>	nl2br(str_replace(array('<', '>'), array('&lt;', '&gt;'), $row->message)),
				'p_edit'	=>	isset($new_comment) && $new_comment == $row->id,
				'p_delete'	=>	$this->user->authenticated() && ($this->user->level >= 3 || $this->user->id == $row->user_id)
			);
		}
		
		return $this->json($comments);
	}
	
	public function submission()
	{
		$parameters = $_POST;
		
		if(!$this->user->authenticated() || !isset($parameters['signature']))
		{
			exit;
		}
		
		unset($parameters['signature']);
		
		ksort($parameters);
		
		$signature = array();
		
		foreach($parameters as $key=>$value)
		{
			if($value === 'null')
			{
				$value = '';
			}
			
			$signature[] = $key.'='.rawurlencode($value);
		}
		
		if(md5(implode('&', $signature).\Pawn::setting('Game.publisher_secret_key')) == $_POST['signature'])
		{
			$game_id = \model\Game::get('WHERE tag=?', $_POST['gameID'])->fetch()->id;
			
			if($_POST['submission'] == 'score')
			{
				$score = $_POST['score'];
				$type = $_POST['datatype'];
				$order = $_POST['sortOrder'];
				
				$select = ($order == 'asc')?'MIN(`score`)':'MAX(`score`)';
				
				$best = db::query('SELECT '.$select.' FROM `'.\model\Score::table_name().'` WHERE `user_id`=? AND `game_id`=?', array($this->user->id, $game_id))->fetchColumn();
				
				$submission = new \model\Score(array
				(
					'user_id'	=>	$this->user->id,
					'game_id'	=>	$game_id,
					
					'score'		=>	$score,
					'type'		=>	$type,
					'order'		=>	$order
				), true);
				
				$submission->create();
				
				if($best === null || ($order == 'asc' && $score < $best) || ($order == 'desc' && $score > $best))
				{
					$users = array();
					$points = 0;
					
					$leaderboard = db::query('SELECT `user_id`, '.$select.' FROM `'.\model\Score::table_name().'` WHERE `game_id`=? AND `user_id`!=? GROUP BY `user_id`', array($game_id, $this->user->id));
					
					foreach($leaderboard as $row)
					{
						if($best === null)
						{
							if(($order == 'asc' && $row[$select] < $score) || ($order == 'desc' && $row[$select] > $score))
							{
								$users[] = $row['user_id'];
							} else
							{
								$points++;
							}
						} else
						{
							if(($order == 'asc' && $row[$select] > $score) || ($order == 'desc' && $row[$select] < $score))
							{
								if(($order == 'asc' && $row[$select] < $best) || ($order == 'desc' && $row[$select] > $best))
								{
									$points++;
								}
							}
						}
					}
					
					$time = time();
					
					if($best === null)
					{
						$points++;
						
						if($users)
						{
							db::query('UPDATE `'.\model\User::table_name().'` SET `points`=`points`+1 WHERE `id` IN ('.implode(',', $users).')');
							db::query('INSERT INTO `'.\model\Points::table_name().'` (`trash`, `ts_create`, `ts_edit`, `user_id`, `points`) VALUES (0, '.$time.', '.$time.', '.implode(', 1), (0, '.$time.', '.$time.', ', $users).', 1)');
						}
					}
					
					$this->user->awardPoints($points);
				}
				
				return $this->json(array
				(
					'exp_level'			=>	$this->user->exp_level,
					'experience'		=>	$this->user->experience(),
					'max_experience'	=>	$this->user->max_experience()
				));
			} else
			{
				$medal = new \model\Achievement(array
				(
					'user_id'		=>	$this->user->id,
					'game_id'		=>	$game_id,
					
					'name'			=>	$_POST['name'],
					'kind'			=>	$_POST['kind'],
					'description'	=>	$_POST['description'],
					'thumbnail'		=>	$_POST['thumbnail']
				), true);
				
				$medal->create();
				
				return $this->json(true);
			}
		}
	}
	
	private function places($game_id, $order=null)
	{
		if(!$this->user->authenticated())
		{
			return false;
		}
		
		if(!isset($order))
		{
			if(!($order = db::query('SELECT `order` FROM `'.\model\Score::table_name().'` WHERE `game_id`=? LIMIT 1', $game_id)->fetchColumn()))
			{
				return false;
			}
		}
		
		$places = array();
		
		$time = time();
		$query = 'SELECT COUNT(*) FROM `'.\model\Score::table_name().'` WHERE `game_id`=? AND `user_id`!=?';
		$compare = ' GROUP BY `user_id` '.(($order == 'asc')?'HAVING MIN(`score`)<?':'HAVING MAX(`score`)>?');
		
		$scores = array
		(
			'day'	=>	86400,
			'week'	=>	604800,
			'month'	=>	2629743,
			'all'	=>	0
		);
		
		foreach($scores as $period=>$length)
		{
			if($score = db::query('SELECT '.(($order == 'asc')?'MIN':'MAX').'(`score`) FROM `'.\model\Score::table_name().'` WHERE `user_id`=? AND `game_id`=?', array($this->user->id, $game_id))->fetchColumn())
			{
				if($length)
				{
					$place = db::query($query.' AND `ts_create`>?'.$compare, array($game_id, $this->user->id, $time - $length, $score))->fetchColumn() + 1;
				} else
				{
					$place = db::query($query.$compare, array($game_id, $this->user->id, $score))->fetchColumn() + 1;
				}
				
				$places[$period] = array
				(
					'score'	=>	$score,
					'place'	=>	$place
				);
			}
		}
		
		return $places;
	}
	
	public function fetch($secret)
	{
		if($secret != \Pawn::setting('Game.publisher_secret_key')) exit;
		
		$game_tag = $_POST['game_tag'];
		
		$contents = \Pawn::remote_contents('http://feedmonger.mochimedia.com/feeds/query/?q=%28game%3A'.$game_tag.'%29&pubid='.\Pawn::setting('Game.publisher_id').'&format=json');
		$request = json_decode($contents);
		
		foreach($request->games as $game)
		{
			$instance = new \model\Game(array('obj'=>$game), true);
			$instance->create();
		}
	}
}